/*
 * Copyright European Commission's
 * Taxation and Customs Union Directorate-General (DG TAXUD).
 */
package eu.europa.ec.taxud.cesop.writers;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.Map.Entry;

import eu.europa.ec.taxud.cesop.utils.ValidationConstants.XML;

import static eu.europa.ec.taxud.cesop.utils.LangUtils.isNotBlank;

/**
 * Utils class to write XML files.
 */
public class CesopXmlWriter<T extends OutputStream> implements AutoCloseable {

    private static final String INDENT_CHARS = "    ";

    private final T outputStream;
    private final XMLStreamWriter xmlStreamWriter;
    int indent = 0;
    boolean onEol = false;

    /**
     * Constructor.
     *
     * @param outputStream the output stream
     * @throws XMLStreamException in case of exception
     */
    public CesopXmlWriter(final T outputStream) throws XMLStreamException {
        final XMLOutputFactory xmlOutputFactory = XMLOutputFactory.newFactory();
        this.outputStream = outputStream;
        this.xmlStreamWriter = xmlOutputFactory.createXMLStreamWriter(outputStream, StandardCharsets.UTF_8.name());
    }

    protected void writeNamespace() throws XMLStreamException {
        this.xmlStreamWriter.writeNamespace(XML.CESOP_XML_NAMESPACE_PREFIX, XML.CESOP_XML_NAMESPACE);
    }

    protected void writeStartDocument() throws XMLStreamException {
        this.xmlStreamWriter.writeStartDocument();
    }

    protected void writeTag(final QName qName, final String value) throws XMLStreamException {
        this.writeTagWithAttributes(qName, value, null);
    }

    protected void writeAttribute(final String name, final String value) throws XMLStreamException {
        this.xmlStreamWriter.writeAttribute(name, value);
    }

    protected void writeTagWithAttributes(final QName qName, final String value, final Map<String, String> attributes) throws XMLStreamException {
        this.writeStartElement(qName);
        if (attributes != null && !attributes.isEmpty()) {
            for (final Entry<String, String> attribute : attributes.entrySet()) {
                this.xmlStreamWriter.writeAttribute(attribute.getKey(), attribute.getValue());
            }
        }
        this.writeText(value);
        this.writeEndElement();
        this.writeEOL();
    }

    protected void writeTag(final QName qName, final boolean value) throws XMLStreamException {
        this.writeStartElement(qName);
        this.writeBoolean(value);
        this.writeEndElement();
        this.writeEOL();
    }

    protected void writeTag(final QName qName, final Integer value) throws XMLStreamException {
        this.writeStartElement(qName);
        this.writeInteger(value);
        this.writeEndElement();
        this.writeEOL();
    }

    protected void writeTagIfNotEmpty(final String localName, final String value) throws XMLStreamException {
        if (isNotBlank(value)) {
            this.writeTag(localName, value);
        }
    }

    protected void writeTagIfNotEmpty(final String localName, final Integer value) throws XMLStreamException {
        if (value != null) {
            this.writeTag(localName, value);
        }
    }

    protected void writeTag(final String localName, final String value) throws XMLStreamException {
        this.writeStartElement(localName);
        this.writeText(value);
        this.writeEndElement();
        this.writeEOL();
    }

    protected void writeTag(final String localName, final Integer value) throws XMLStreamException {
        this.writeStartElement(localName);
        this.writeInteger(value);
        this.writeEndElement();
        this.writeEOL();
    }

    protected void writeTag(final String localName, final Long value) throws XMLStreamException {
        this.writeStartElement(localName);
        this.writeLong(value);
        this.writeEndElement();
        this.writeEOL();
    }

    protected void writeStartElement(final QName qName) throws XMLStreamException {
        this.writeIndentIfNeeded();
        this.xmlStreamWriter.writeStartElement(qName.getPrefix(), qName.getLocalPart(), qName.getNamespaceURI());
        this.indent++;
    }

    protected void writeStartElement(final String localName) throws XMLStreamException {
        this.writeIndentIfNeeded();
        this.xmlStreamWriter.writeStartElement(localName);
        this.indent++;
    }

    protected void writeEndElement() throws XMLStreamException {
        this.indent--;
        this.writeIndentIfNeeded();
        this.xmlStreamWriter.writeEndElement();
    }

    protected void writeText(final String characters) throws XMLStreamException {
        this.xmlStreamWriter.writeCharacters(characters);
    }

    protected void writeBoolean(final boolean value) throws XMLStreamException {
        this.xmlStreamWriter.writeCharacters(Boolean.toString(value));
    }

    protected void writeInteger(final Integer value) throws XMLStreamException {
        this.xmlStreamWriter.writeCharacters(Integer.toString(value));
    }

    protected void writeLong(final Long value) throws XMLStreamException {
        this.xmlStreamWriter.writeCharacters(Long.toString(value));
    }

    protected void writeEOL() throws XMLStreamException {
        this.xmlStreamWriter.writeCharacters("\n");
        this.onEol = true;
    }

    protected void writeIndentIfNeeded() throws XMLStreamException {
        if (!this.onEol) {
            return;
        }
        for (int i = 0; i < this.indent; i++) {
            this.xmlStreamWriter.writeCharacters(INDENT_CHARS);
        }
        this.onEol = false;
    }

    protected void flush() throws XMLStreamException {
        this.xmlStreamWriter.flush();
    }

    /**
     * Returns the output stream.
     *
     * @return the output stream.
     */
    public T getOutputStream() {
        return this.outputStream;
    }

    @Override
    public void close() throws XMLStreamException, IOException {
        this.xmlStreamWriter.close();
        this.outputStream.close();
    }
}
